----------------------------------------------------------------------------------------------------
-- Animated Map Objects (V1.0.0)
----------------------------------------------------------------------------------------------------
-- Purpose:  Create animated map objects
-- Authors:  Schluppe
--
-- Copyright (c) none - free to use or modify, 2021
----------------------------------------------------------------------------------------------------
-- Purpose: This scripts enables you to use the animated map objects as known in LS19
--
-- How to use:
-- - Add a reference to the script to your modDesc.xml (extraSourceFiles)
-- - Add the script and proper settings to an object within the scene (e.g. transformation group)
--	 - Add a new attribute called 'onCreate', with type 'Script Callback', 
--	    and set it to 'modOnCreate.AnimatedMapObject'
--	 - Add another new attribute called 'index', with type 'string'.  
--		The value does define the name of the element and must be unique in your map. 
--	 - Add another new attribute called 'xmlFilename', with type 'string'.  
--		The value does define the relative path to the 'animatedObjects.xml' file. 
--
-- Example:
--	Map.I3D
-- 		<Attribute name="index" type="string" value="swingGate06"/>
-- 		<Attribute name="onCreate" type="scriptCallback" value="modOnCreate.AnimatedMapObject"/>
-- 		<Attribute name="xmlFilename" type="string" value="maps/xml/animatedObjects.xml"/>
--
--  animatedObjects.xml
--		<animatedObject index="swingGate06">
--			<controls posAction="ACTIVATE_HANDTOOL" triggerNode="0" posText="action_openGate" negText="action_closeGate"/>
--			<animation duration="3">
--				<part node="1|0|0">
--					<keyFrame time="0.0" rotation="0 0 0"/>
--					<keyFrame time="0.9" rotation="0 87.295 0"/>
--				</part>
--				<part node="2|0|0">
--					<keyFrame time="0.1" rotation="0 0 0"/>
--					<keyFrame time="1" rotation="0 -86.845 0"/>
--				</part>
--			</animation>
--			<sounds>
--				<moving file="$data/sounds/prefab/gate/gateLoop.wav" loops="0" linkNode="1" volume="0.8" radius="25" innerRadius="3" fadeOut="0.25"/>
--				<posEnd file="$data/sounds/prefab/gate/gateOpened.wav" linkNode="1" volume="0.8" radius="25" innerRadius="3"/>
--				<negEnd file="$data/sounds/prefab/gate/gateClosed.wav" linkNode="1" volume="0.8" radius="25" innerRadius="3"/>
--			</sounds>
--		 </animatedObject>
--
----------------------------------------------------------------------------------------------------

animatedMapObjects = {}
local animatedMapObjects_mt = Class(animatedMapObjects, AnimatedObject)

--- Entry point for the class, as declared in the user script 
function animatedMapObjects.onCreate(id)
	local self = animatedMapObjects.new(g_server ~= nil, g_client ~= nil)
	animatedMapObjects:createXMLSchema()

	if animatedMapObjects:load(id, self) then
		g_currentMission:addOnCreateLoadedObject(self)
		g_currentMission:addOnCreateLoadedObjectToSave(self)
		self:register(true)
	else
		self:delete()
	end
end

--- Create a new instance of the AnimatedMapObjects
function animatedMapObjects:new(isServer, isClient, customMt)
	local self = AnimatedObject.new(isServer, isClient, customMt or animatedMapObjects_mt)
	return self
end

--- Deletes the instance of the AnimatedMapObjects => Unsubscribe Events 
function animatedMapObjects:delete()
	g_currentMission:removeOnCreateLoadedObjectToSave(self)
	animatedMapObjects:superClass().delete(self)
end

function animatedMapObjects:loadFromXMLFile(xmlFile, key)
	local animTime = getXMLFloat(xmlFile, key .. "#time")
	if animTime ~= nil then
		self.animation.direction = getXMLInt(xmlFile, key .. "#direction")
		self:setAnimTime(animTime, true)
	end

	AnimatedObject.hourChanged(self)

	return true
end

function animatedMapObjects:saveToXMLFile(xmlFile, key, usedModNames)
	setXMLFloat(xmlFile, key .. "#time", self.animation.time)
	setXMLInt(xmlFile, key .. "#direction", self.animation.direction)
end

function animatedMapObjects:load(nodeId, self)
	local xmlFilename = getUserAttribute(nodeId, "xmlFilename")
	if xmlFilename == nil then
		print("Error: Missing 'xmlFilename' user attribute for animatedMapObjects node '" .. getName(nodeId) .. "'!")
		return false
	end

	local baseDir = g_currentMission.loadingMapBaseDirectory
	if baseDir == "" then
		baseDir = Utils.getNoNil(self.baseDirectory, baseDir)
	end

	xmlFilename = Utils.getFilename(xmlFilename, baseDir)
	local index = getUserAttribute(nodeId, "index")

	if index == nil then
		print("Error: Missing 'index' user attribute for animatedMapObjects node '" .. getName(nodeId) .. "'!")
		return false
	end

	local xmlFileId = loadXMLFile("AnimatedObject", xmlFilename)
	if xmlFileId == 0 then
		return false
	end

	local key = nil
	local i = 0
	while true do
		local objectKey = string.format("animatedObjects.animatedObject(%d)", i)
		if not hasXMLProperty(xmlFileId, objectKey) then
			break
		end

		local configIndex = getXMLString(xmlFileId, objectKey .. "#index")
		if configIndex == index then
			key = objectKey
			break
		end

		i = i + 1
	end

	if key == nil then
		print("Error: index '" .. index .. "' not found in AnimatedObject xml '" .. xmlFilename .. "'!")
		return false
	end

	local xmlFile = XMLFile.load("AnimatedObject", xmlFilename, animatedMapObjects.xmlSchema)
	local triggerId = xmlFile:getValue(key .. ".controls#triggerNode", nil, nodeId)
	local posText = xmlFile:getValue(key .. ".controls#posText")
	local result = animatedMapObjects:superClass().load(self, nodeId, xmlFile, key, xmlFilename, i3dMappings)

	delete(xmlFileId)

	return result
end

function animatedMapObjects:createXMLSchema()
	if animatedMapObjects.xmlSchema == nil then
		local schema = XMLSchema.new("AnimatedMapObject")
		animatedMapObjects:superClass().registerXMLPaths(schema, "animatedObjects")		
		animatedMapObjects.xmlSchema = schema
	end
end

--- Registration
g_onCreateUtil.addOnCreateFunction("AnimatedMapObject", animatedMapObjects.onCreate)
